/*************************************************************************************************
// INTEL CONFIDENTIAL Copyright 2011-2016 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to the source code
// ("Material") are owned by Intel Corporation or its suppliers or licensors. Title to the Material
// remains with Intel Corporation or its suppliers and licensors. The Material contains trade secrets
// and proprietary and confidential information of Intel or its suppliers and licensors. The Material is
// protected by worldwide copyright and trade secret laws and treaty provisions. No part of the
// Material may be used, copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual property right is
// granted to or conferred upon you by disclosure or delivery of the Materials, either expressly, by
// implication, inducement, estoppel or otherwise. Any license under such intellectual property
// rights must be express and approved by Intel in writing.
//*************************************************************************************************/

#ifndef __SS_I2C_H_
#define __SS_I2C_H_

#include "dsdt.h"

#define DATA_BYTE_MAX               32    /*! i2c max data bytes*/
#define ADDR_BYTES_MAX              4     /*! i2c max addreass bytes*/
#define SPB_TRANSFERS               3     /*! i2c spb transfers numbers*/
#define SPB_WRITE_TRANSFERS         1     /*! i2c spb transfers write numbers*/

/*! "brief DATA_TOKE_TYPE
 *    enum of DATA_TOKE_TYPE
 */
typedef enum _E_REGS_TOKE_TYPE__
{
    DATA_8BITS =  0x0001,
    DATA_16BITS = 0x0002,
    DATA_24BITS = 0x0003,
    DATA_32BITS = 0x0004,
    DATA_TOKE_DELAY = 0x0020,
    DATA_TOKE_TERM = 0xffff
} DATA_TOKE_TYPE;

/*! "brief ADDR_TOKE_TYPE
 *    enum of ADDR_TOKE_TYPE
 */
typedef enum _E_REGS_ADDR_TYPE__
{
    ADDR_8BITS  = 0x0001,
    ADDR_16BITS = 0x0002,
    ADDR_24BITS = 0x0003,
    ADDR_32BITS = 0x0004,
} ADDR_TOKE_TYPE;

/*! "brief I2C_RESOURCE
 *    data structure of i2c resource
 */
typedef struct I2C_RESOURCE__
{
    LARGE_INTEGER           ConnectionId; /*! i2c wdf connection id*/
    WDFIOTARGET             Target;       /*! i2c wdf target*/
    UINT8                   Enable;       /*! i2c enabling or not */
    UINT8                   Value;        /*! i2c slave address*/
    UINT8                   Flag;         /*! i2c supporting*/
    UINT8                   Bus;          /*! i2c bus*/
    ADDR_TOKE_TYPE          AddrBits;     /*! i2c address bits*/
} I2C_RESOURCE, *PI2C_RESOURCE;

/*! "brief I2C_CONTEXT
 *    data structure of i2c context
 */
typedef struct I2C_CONTEXT__
{
    WDFDEVICE     DevHandle;              /*! sensor device handle*/
    const char*   DevName;                /*! sensor device name*/
    I2C_RESOURCE  I2c[I2C_NUM];           /*! i2c resource table*/
} I2C_CONTEXT, *PI2C_CONTEXT;

/*! "brief SsI2cZeroInit
 *    I2c context init with zero status
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    WDFDEVICE hDevice - sensor device handle
 *    const char* DeviceName - sensor device name
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS SsI2cZeroInit(
    IN PI2C_CONTEXT pI2cDev,
    IN WDFDEVICE hDevice,
    IN const char* DeviceName
    );

/*! "brief SsI2cSetAddrBits
 *    I2c set address bits type
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    ULONG Index - i2c slave address index
 *    ADDR_TOKE_TYPE Value - address bits type
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS SsI2cSetAddrBits(
    IN PI2C_CONTEXT pI2cDev,
    IN ULONG Index,
    ADDR_TOKE_TYPE Value
    );

/*! "brief SsI2cSlaveInit
 *    I2c slave address initialization
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    ULONG Index - i2c slave address index
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS SsI2cSlaveInit(
    IN PI2C_CONTEXT pI2cDev,
    IN ULONG Index
    );

/*! "brief SsI2cSlaveUninit
 *    I2c slave address un-initialization
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    ULONG Index - i2c slave address index
 * Return:
 *    NTSTATUS - status
 */
void SsI2cSlaveUninit(
    IN PI2C_CONTEXT pI2cDev,
    IN ULONG Index
    );

/*! "brief i2cWrite
 *    i2c write
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    UINT16 Addr - register address
 *    ULONG Data - register data to write
 *    DATA_TOKE_TYPE DataBits - data bits type
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS i2cWrite(
    PI2C_CONTEXT pI2cDev,
    UINT16 Addr,
    ULONG Data,
    DATA_TOKE_TYPE DataBits
    );

/*! "brief i2cRead
 *    i2c read
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    UINT16 Addr - register address
 *    ULONG* Data - pointer of data to read
 *    DATA_TOKE_TYPE DataBits - data bits type
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS i2cRead(
    PI2C_CONTEXT pI2cDev,
    UINT16 Addr,
    ULONG *Data,
    DATA_TOKE_TYPE DataBits
    );

/*! "brief RegReadBuf
 *    i2c read buffer
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    UINT16 Addr - register address
 *    UINT8 *Data - pointer of data to read
 *    ULONG Length - data length
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS RegReadBuf(
    PI2C_CONTEXT pI2cDev,
    UINT16 Addr,
    UINT8 *Data,
    ULONG Length
    );

/*! "brief RegWriteBuf
 *    i2c write buffer
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    UINT8 *Data - pointer of data to write
 *    ULONG Length - data length
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS RegWriteBuf(
    PI2C_CONTEXT pI2cDev,
    UINT8* Data,
    ULONG Length
    );

/*! "brief i2cWriteExt
 *    i2c write
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    UINT16 Addr - register address
 *    ULONG Data - register data to write
 *    DATA_TOKE_TYPE DataBits - data bits type
 *    ADDR_TOKE_TYPE AddrBits - address bits type
 *    ULONG I2CIndex - I2C target index
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS i2cWriteExt(
    PI2C_CONTEXT pI2cDev,
    ULONG Addr,
    ULONG Data,
    DATA_TOKE_TYPE DataBits,
    ADDR_TOKE_TYPE AddrBits,
    ULONG I2CIndex
    );

/*! "brief i2cReadExt
 *    i2c read
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    UINT16 Addr - register address
 *    ULONG* Data - pointer of data to read
 *    DATA_TOKE_TYPE DataBits - data bits type
 *    ADDR_TOKE_TYPE AddrBits - address bits type
 *    ULONG I2CIndex - I2C target index
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS i2cReadExt(
    PI2C_CONTEXT pI2cDev,
    ULONG Addr,
    ULONG *Data,
    DATA_TOKE_TYPE DataBits,
    ADDR_TOKE_TYPE AddrBits,
    ULONG I2CIndex
    );

/*! "brief RegReadBufExt
 *    i2c read buffer
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    UINT16 Addr - register address
 *    UINT8 *Data - pointer of data to read
 *    ULONG Length - data length
 *    ADDR_TOKE_TYPE AddrBits - address bits type
 *    ULONG I2CIndex - I2C target index
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS RegReadBufExt(
    PI2C_CONTEXT pI2cDev,
    ULONG Addr,
    UINT8 *Data,
    ULONG Length,
    ADDR_TOKE_TYPE AddrBits,
    ULONG I2CIndex
    );

/*! "brief RegWriteBufExt
 *   i2c write buffer
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    UINT8 *Data - pointer of data to write
 *    ULONG Length - data length
 *    ULONG I2CIndex - I2C target index
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS RegWriteBufExt(
    PI2C_CONTEXT pI2cDev,
    UINT8* Data,
    ULONG Length,
    ULONG I2CIndex
    );

/*! "brief i2cW
 *    i2c write
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    ULONG Addr - register address
 *    ULONG Data - register data to write
 *    ADDR_TOKE_TYPE AddrBits - address bits type
 *    DATA_TOKE_TYPE DataBits - data bits type
 *    ULONG Index - i2c slave address index
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS i2cW(
    PI2C_CONTEXT pI2cDev,
    ULONG Addr,
    ULONG Data,
    ADDR_TOKE_TYPE AddrBits,
    DATA_TOKE_TYPE DataBits,
    ULONG Index
    );

/*! "brief i2cR
 *    i2c read
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    ULONG Addr - register address
 *    ULONG *Data - register data to read
 *    ADDR_TOKE_TYPE AddrBits - address bits type
 *    DATA_TOKE_TYPE DataBits - data bits type
 *    ULONG Index - i2c slave address index
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS i2cR(
    PI2C_CONTEXT pI2cDev,
    ULONG Addr,
    ULONG *Data,
    ADDR_TOKE_TYPE AddrBits,
    DATA_TOKE_TYPE DataBits,
    ULONG Index
    );

/*! "brief i2cWb
 *    i2c write
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    UINT8* Data - register data to write
 *    ULONG Length - data length to write
 *    ULONG Index - i2c slave address index
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS i2cWb(
    PI2C_CONTEXT pI2cDev,
    UINT8* Data,
    ULONG Length,
    ULONG Index
    );

/*! "brief i2cRb
 *    i2c read
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    ULONG Addr - register address
 *    UINT8 *Data - register data to read
 *    ADDR_TOKE_TYPE AddrBits - address bits type
 *    ULONG Length - data length to read
 *    ULONG Index - i2c slave address index
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS i2cRb(
    PI2C_CONTEXT pI2cDev,
    ULONG Addr,
    UINT8 *Data,
    ADDR_TOKE_TYPE AddrBits,
    ULONG Length,
    ULONG Index
    );

/*! "brief i2cVcmR1
 *    i2c read
 *
 * Arg:
 *    PI2C_CONTEXT pI2cDev - context pointer of i2c
 *    UINT16 Addr - register address
 *    ULONG Index - i2c slave address index
 * Return:
 *    NTSTATUS - status
 */
NTSTATUS i2cVcmR1(
    PI2C_CONTEXT pI2cDev,
    UINT16 *Data,
    ULONG Index
    );

#endif
