﻿// INTEL CONFIDENTIAL
// Copyright 2014-2016 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to the source code 
// ("Material") are owned by Intel Corporation or its suppliers or licensors. Title to the 
// Material remains with Intel Corporation or its suppliers and licensors. The Material 
// contains trade secrets and proprietary and confidential information of Intel or its 
// suppliers and licensors. The Material is protected by worldwide copyright and trade 
// secret laws and treaty provisions. No part of the Material may be used, copied,
// reproduced, modified, published, uploaded, posted, transmitted, distributed, or 
// disclosed in any way without Intel’s prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual property right 
// is granted to or conferred upon you by disclosure or delivery of the Materials, either 
// expressly, by implication, inducement, estoppel or otherwise. Any license under such 
// intellectual property rights must be express and approved by Intel in writing.
//
// Module Name:
//
//    SensorDataInterface.h
//
// Abstract:
//
//    This module contains the common declarations shared by drivers
//
// Environment:
//
//    kernel

#ifndef _SENSOR_DATA_INTERFACE_H
#define _SENSOR_DATA_INTERFACE_H

#include "InterfacesCommon.h"
#include "public.h"
#include "VTIO_BIOS_Manifest.h"

typedef PHYSICAL_ADDRESS FBPT_PA, *PFBPT_PA;
typedef PUCHAR FBPT_VA, *PFBPT_VA;

typedef struct _CIO2_FBPT_DESCRIPTOR
{
    OUT FBPT_VA	*FrameFBPT_VA;	// Virtual address of the current FBPT entry of the reported frame. This field is useful ONLY in the case the frame report
    // mode is FBRM_FRAMES_BUNDLE_LINE_NUMBER. This pointer allows the AVStream driver to provide the ISP with the address of the FBPT
    // to allow the ISP FWE to do progress on frame pixels up to the current received line number (in order not to process not relevant pixels).
    // At the end of ISP processing of this frame, the AVStream driver is supposed to call the CompletedFramesBundleProcessing API of the CSI2 host
    // controller driver which in turn will return the completed frame buffer with the latest updated status to the AVStream driver by calling
    // the FRAME_COMLETE callback (now with State of FRAME_REPORT_END_OF_FRAME).
    // At this point the ownership of IMAGE_FRAMES_BUNDLE of the completed frame returns to the AVStream driver and the CurrentFrameFBPT_VA
    // is not any more relevant for the AVStream driver!
    OUT FBPT_PA	*FrameFBPT_PA;	// Physical address of the current FBPT entry of the reported frame. This field is useful ONLY in the case the frame report
    // mode is FBRM_FRAMES_BUNDLE_LINE_NUMBER. This pointer allows the AVStream driver to provide the ISP with the address of the FBPT
    // to allow the ISP FWE to do progress on frame pixels up to the current received line number (in order not to process not relevant pixels).
    // At the end of ISP processing of this frame, the AVStream driver is supposed to call the CompletedFramesBundleProcessing API of the CSI2 host
    // controller driver which in turn will return the completed frame buffer with the latest updated status to the AVStream driver by calling
    // the FRAME_COMLETE callback (now with State of FRAME_REPORT_END_OF_FRAME).
    // At this point the ownership of IMAGE_FRAMES_BUNDLE of the completed frame returns to the AVStream driver and the CurrentFrameFBPT_PA
    // is not any more relevant for the AVStream driver!
} CIO2_FBPT_DESCRIPTOR, *PCIO2_FBPT_DESCRIPTOR;


typedef struct _CAPTURE_BUFFER
{
    // Note: if a specific data type is not used at a created stream (UNUSED_DATA_TYPE), the elements in this structure are not used (the below BufferMDL is NULL).

    IN PMDL		BufferMDL;		// Provided by AVStream driver. In case of a NULL pointer - buffer is not available.
    // Note that in CSI2, Post Image Data has same data type as of pre image data and hence such data will go into
    // same buffer.

#ifdef SECURE_INPUT_FRAME
                                // VTIO
    IN PGUID    BufferGUID;     // Provided by AVStream driver. In case of a NULL pointer - buffer is not available.
                                // Note that in CSI2, Post Image Data has same data type as of pre image data and hence such data will go into
                                // same buffer.
                                // Note: BufferGUID is used only when this link's SetMipiLinkSecurityState is LINK_SECURED
                                //       BufferGUID is used by streams of secured cameras where their memory allocation is in VTL1.
#else
    IN ULONG	BufferHandle;	// If NULL, means that the client driver
    // (SkyCam AVStream) does not make reuse of the buffer that is described
    // in this descriptor (e.g. buffer that is allocated by the OS for SoC
    // sensors and its allocation can change).
    // If not NULL, the value is a unique handle for the buffer
    // provided by this descriptor. It is a hint for the CSI2 host controller
    // driver that this buffer is recycled (like ping pong buffer)
    // between the client driver (SkyCam AVStream driver) and the CSI2 host
    // controller driver. The CSI2 host controller driver typically translates
    // the given MDL to host controller LOP pages in its FBPT. This hint can save the
    // driver repeated allocation and deallocation of such LOP buffer and
    // save it the need to refill the LOP buffers from the provided MDL and
    // this way to cache MDLs ready to use. Whenever the AVStream driver sends
    // a buffer that was already used in the past by the CSI2 host controller
    // to be re filled, it would add to the buffer submission its BufferHandle
    // that will be used by the CSI2 host controller as a key to its cached
    // LOPs.
    // Note: if the BufferHandle is not NULL, the CIO2 driver must track those LOP that are
    // saved for reuse based on the BufferHandle key, and when the stream is
    // stopped (Stop commnad, or Data Stream interface is dereferenced), the CIO2 driver
    // must release those LOPs as the buffers are not expected to be reused between two
    // runs of the interface.
#endif

    OUT ULONG	DataLen;				// Actual data length in the BufferMDL


    OUT	CIO2_FBPT_DESCRIPTOR FrameReceiveProgress; // This field is valid only after the CSI2 host controller generated an interrupt on line
    // number and as a result the CSI2 host controller driver reports the partial frame buffer to the AVStream driver with
    // frame status of FRAME_REPORT_REACHED_FRAME_LINE.
    // This field is only useful in FRAME_REPORT_REACHED_FRAME_LINE	mode. This is a pointer to the frame's FBPT entry. The CurLineNum
    // field should be used by the ISP device as a fence to be sure the input buffer is filled up to the next expected chunk that
    // it wants to process.
    //
    // Note: The pointed FBPT entry is already pinned and the AVStream drive should only use it without releasing it at the end of
    // processing.
    //	** >> In addition, I need to think about an optimization here as we do
    //		not want the AVStream driver to map it for every frame into the
    //		ISP's MMU, so a possible option is that the CSI2 HC driver would
    //		provide the whole FBPT table by its physical and virtual address
    //		to the AVStream, and with every frame notification this pointer
    //		here will be replaced by the offset from the FBPT start address
    //		(as all FBPT entries must be contiguous to each other anyway from
    //		the CSI2 HC reasons). See comment with 'INT_ON_LINE_NUM' below -
    //		about this idea.

    OUT ULONG	ErrorStatus;			// Bit field. Suggested errors (the fileds in this ULONG can be extracted from the FBPT's CmpCode status once
        // the frame valid bit has been flipped by the CIO2 host controller. ToDO - TBD):
        // - Buffer Too Short (fully written before reached end of frame). This bit can be set, along with Success
        //	 status bit. 
        // - Success capture
        // - Flash trigger happened at the start of this frame. This bit is set only at the buffer descriptor for which the CIO2 device
        //	 triggered the flash controller. The flash trigger pin was set to the PMIC (flash controller) after the pre programmed delay
        //	 at the CIO2 expired. It is the responsibility of the upper layer SW to analyze each frame following this report bit with the
        //   preprogrammed delays and flash pulse width to know which frame was partially exposed to flash when it started, which frame was
        //   fully exposed to the flash and which frame was was partially exposed to flash when it ended.
        // - Framing errors (CRC) seen - At least one, but less than threshold specified CRC errors seen in buffer stream.
        //	 This is considered as a soft error and when set,
        //	 Success Capture is also set.
        // - Framing error (CRC) detected - corrupted frame. More than threshold specified CRC errors seen in buffer stream
        //	 (Success Capture must be FALSE).

} CAPTURE_BUFFER, *PCAPTURE_BUFFER;

#define FLASH_BIT_FLASH_TRIGGERED (0x800) // bit 11 of the ErrorStatus
#define CIO_DMA_HALT_NOT_AN_ERROR (0x80000000)

// Frame descriptor. Frame is transferred over a single VC.
// Contains set of pointers to set of frame data buffer and meta-data describing the frame (up to 16 per frame, but realistically it is typical 2 for RAW sensors).
typedef struct _IMAGE_FRAME_DESCRIPTOR
{
    UCHAR VC_Value;	// VC used for this stream. Supported values are 0,1,2,3. VC_Value can be only one of those that were defined in CreateSensorDataLink. If
    // SubmitFramesBundle uses an unsupported VC_Value, an error code will be returned.


    CAPTURE_BUFFER Buffer[CIO2_DATA_TYPES_MAX];	// Note: 1 - The enumerator for this structure must be with same indexes as of the VC_SETTINGS at CreateSensorDataLink.
        //		 2 - If a specific data type is not used at a created stream (UNUSED_DATA_TYPE), its element in this structure is not used.
} IMAGE_FRAME_DESCRIPTOR, *PIMAGE_FRAME_DESCRIPTOR;

// Image frames bundle
// -------------------
// Image frames bundle represents multiple frames that each is transferred over a different VC, in SAME frame rate.
// When more than one bundle entry is used, only one of them will specify ExpectedLastDataTypeBeforeEOF in one of its buffers
// to allow generation of interrupt after all buffers from all related VCs completed their capture. 
// A classical RAW sensor would use a single bundle entry.
// A RAW sensor that transmits HDR frmaes in 3 different VCs in SAME frame rate would utilize 3 bundle entries.
// An SoC sensor that transmits preview image and stream image in two different VCs in same frame rate would use 2 bundle entries.

// Contains set of pointers to set of frame data buffer and meta-data describing the frame (up to 16 per frame, but realistically it is typical 2 for RAW sensors).
typedef struct _IMAGE_FRAMES_BUNDLE
{
    IMAGE_FRAME_DESCRIPTOR ImageFrameDescriptor[CIO2_VC_NUMBER]; // When a single VC is used, the ImageFrameDescriptor[x].Buffer[y].BufferMDL of the unused VCs is NULL

    IN BOOLEAN	NotifyOnEvent;	// This flag is related to the selected ReportMode
    // (See FRAMES_BUNDLE_REPORT_MODE below). When TRUE, AVStream driver asks CSI2 HC
    // driver to generate an interrupt for this frames bundle (on the selected Report
    // mode - FBRM_START_OF_FRAMES_BUNDLE or 
    // FBRM_END_OF_FRAMES_BUNDLE) and provide frames bundle notification to the
    // AVStream driver through the FramesBundleComplete callback. In case the flag is
    // FALSE then the CSI2 host controller driver does not report this
    // specific IMAGE_FRAMES_BUNDLE and it queues the completed descriptors
    // and their related frames until it reaches an IMAGE_FRAMES_BUNDLE with
    // the NotifyOnEvent flag set. In such case the CSI2 host controller driver
    // reports all queued completed IMAGE_FRAMES_BUNDLEs back to back in
    // their comletion order.
    // In case the AVStream driver choose FBRM_FRAMES_BUNDLE_LINE_NUMBER
    // as the selected Report mode, then NotifyOnEvent must be TRUE for every
    // IMAGE_FRAMES_BUNDLE.

    OUT ULONG	FrameNumber;			// Provided by CSI-2 HCD
    OUT ULONGLONG TimeStamp;			// System time when frames bundle completed. Host
        // controller driver should keep in this field system time as close to the
        // interrupt of this frame.
        // An option for the CSI-2 host controller driver is to get the timestamp
        // value that the host controller HW adds to the frame descriptor and use
        // it to fill this field.

#ifdef SECURE_INPUT_FRAME
    IN void *CachedBundleContext;       // VTIO
                                        // MDL caching context. When the frames bundle is provided 1st time to the CIO2 driver through SubmitFramesBundle API,
                                        // this field must be NULL. It means that the frames bundle buffers are pinned as expected, but are not cached. After
                                        // the CIO2 driver calls the CacheFramesBundle API successfuly, at the next times that the AVStream driver submits this
                                        // frames bundle to the CIO2 driver for capture again through SubmitFramesBundle API, this field must have the value
                                        // returned by the CacheFramesBundle API.
                                        //
                                        // NULL - means that this frame bundle is not yet cached by the AVStream.
                                        //      The CIO2 driver cannot assume this bundle will ever return again, thus when it completes the capture into this
                                        //      bundle, the CIO2 driver must remove it from its FBPT before it returns it to the AVStream driver by calling
                                        //      CompletedFramesBundleProcessing. When reciving aframe bundles with NULL in this field, the CIO2 should call the
                                        //      CacheFramesBundle API with its CachedBundleContext set in this field to allow further re use of the cached context
                                        //      of this frames bundle.
                                        //      The CIO2 may optimize and flag this bundle as not cachable (in a side list / cache) such that in next time it receives
                                        //      same frame bundle it will not call again the CacheFramesBundle callback. Or, call the CacheFramesBundle callback
                                        //      again and again (the overhead is not likely to be big).
                                        // Other - A handle (not NULL). This is the acknowledge that the AVStream cached this frame bundle buffers. AVStream
                                        //      driver will use this value also at the next time that it submits this frame bundle in the CachedBundleContext
                                        //      at the IMAGE_FRAMES_BUNDLE.
                                        //      Once the CIO2 wants to remove the frame bundle from its FBPT, it should first call the ReleaseCacheFramesBundle
                                        //      with this CachedBundleContext value and only then should it call the FramesBundleComplete. Note, if the
                                        //      CachedBundleContext is NULL, then the CIO2 drive must not call the ReleaseCacheFramesBundle callback.
                                        //      When not NULL, the CIO2 should keep the CachedBundleContext and the frames bundle (cache them). Next time that
                                        //      the AVStream calls SubmitFramesBundle, the CIO2 driver can check if there is a match of the CachedBundleContext
                                        //      with one of the cached ones.
                                        //      If match is found then no need to insert to the FBPT. Otherwise - no match, then the CIO2 driver should call
                                        //      for the non match frames bundle the CacheFramesBundle callback.
#endif

    IN PVOID Cookie;	// A per frames bundle context within the AVStream driver. IO driver does not modify this field.
} IMAGE_FRAMES_BUNDLE, *PIMAGE_FRAMES_BUNDLE;

typedef enum
{
    COMPRESSION_NONE,							// 000b - No Compression
    COMPRESSION_10_6_10,						// 001b - Compression Scheme 10-6-10. Not supported by the CIO2 host controller
    COMPRESSION_10_7_10,						// 010b - Compression Scheme 10-7-10. Not supported by the CIO2 host controller
    COMPRESSION_10_8_10,						// 011b - Compression Scheme 10-8-10
    COMPRESSION_12_6_12,						// 100b - Compression Scheme 12-6-12. Not supported by the CIO2 host controller
    COMPRESSION_12_7_12,						// 101b - Compression Scheme 12-7-12. Not supported by the CIO2 host controller
    COMPRESSION_12_8_12,						// 110b - Compression Scheme 12-8-12
    MAX_COMPRESSION								// This value is used for error checks
} COMPRESSION_SCHEME;

typedef enum
{
    PREDICTOR_TYPE_1,							// 0b - Predictor type 1
    PREDICTOR_TYPE_2,							// 1b - Predictor type 2
    MAX_PREDICTOR_TYPE							// This value is used for error checks
} COMPRESSION_PREDICTOR;


// Callback definition for a single frames bundle reporting
typedef NTSTATUS(*FRAMES_BUNDLE_COMPLETE)(
    PVOID AVStreamDriverContext,		// Context provided by driver
    PIMAGE_FRAMES_BUNDLE pFramesBundle,	// Frames bundle received
    FRAME_REPORT_STATE State			// Frames bundle state
    );

typedef struct _DATA_STREAM_INTERFACE *PDATA_STREAM_INTERFACE;

#ifdef SECURE_INPUT_FRAME
// VTIO
// Implemented by the AVStream driver.
// CSI2 host controller driver calls this callback function with its allocated CachedBundleContext in the IMAGE_FRAMES_BUNDLE when it receives a frame bundle that
// is not yet cached (e.g. the CachedBundleContext at the received IMAGE_FRAMES_BUNDLE is NULL).
// Returned STATUS may be either:
//      - SUCCESS: CachedBundleContext field in the provided pFramesBundle is accepted by the AVStream as expected. All bundle buffers are cached bu the AVStream.
//              CIO2 driver can perform its own caching now. Frames bundle buffers are guarnateed to remain pinned until the stream is closed (at this point the
//              CIO2 should call ReleaseCacheFramesBundle), or until the AVStream would call the CancelCacheFramesBundle. No field in the provided pFramesBundle
//              is changed by the AVStream driver (it copies the CachedBundleContext from the provided IMAGE_FRAMES_BUNDLE to its store of the frames bundle).
//        Note: In case of one of the below FAIL statuses, the CachedBundleContext field in the provided pFramesBundle is ignored by the AVStream driver.
//      - FAIL_BAD_PARAM: One of the parameters is not as expected.
//      - FAIL_CACHING_IS_DISABLED: When AVStream driver does not support MDL caching 
//      - FAIL_BUNDLE_IS_NOT_CACHED: When the AVStream failed to cache all of the bundle buffers.
//
// Note: If the CIO2 driver does not call the CacheFramesBundle API in the AVStream, or in case the call to CachedBundleContext result with FAIL status, it cannot
//       assume that the frame bundle will ever return again to the CIO2 driver after it is returned to the AVStream. In such case the CIO2 driver behaves in the
//       legacy mode and inserts the buffers to the FBPT before it provides it to the CIO2 device, and at the end of capture it must remove it from the FBPT before
//       it returns it to the AVStream.
typedef NTSTATUS(*CACHE_FRAMES_BUNDLE)(
    PVOID AVStreamDriverContext,        // Context provided by the AVStream driver
    PIMAGE_FRAMES_BUNDLE pFramesBundle  // Frames bundle received at SubmitFramesBundle
);

// VTIO
// Implemented by the AVStream driver.
// CSI2 host controller driver calls this callback function when it stops using a frame bundle that was cached by prior call to CacheFramesBundle.
// CIO2 stops using frame bundles when the stream is stopped, or, when the AVStream driver calls the cancellation routin if the MF did similar call to the
// AVStream for the MDL caching for AVStream buffer that is sent to the CIO2 (this is only possible if the stream is of an SoC sensor, otherwise the CIO2
// capture buffers are allocated in the AVStream driver and these type of buffers are not likely be cancelled.
// CachedBundleContext in the pFramesBundle is the value that the CIO2 provided at the call to CacheFramesBundle callback function.
// Calling this function when CachedBundleContext in the pFramesBundle is NULL is not allowed!
// Returned STATUS may be either:
//      - SUCCESS: The frames bundle buffers are not cached anymore by the AVStream driver for the CIO2 driver.
//              When returned status is SUCCESS, then the CachedBundleContext in pFramesBundle is written by the AVStream as NULL. No other
//              field in the provided pFramesBundle is changed by the AVStream driver.
//      - FAIL_BAD_PARAM: One of the parameters is not as expected.
//      - FAIL_CACHING_IS_DISABLED: When AVStream driver does not support MDL caching
//      - FAIL_BUNDLE_IS_NOT_CACHED: This frames bundle is not marked at the AVStream driver as cached.
typedef NTSTATUS(*RELEASE_CACHE_FRAMES_BUNDLE)(
    PVOID AVStreamDriverContext,        // Context provided by the AVStream driver
    PIMAGE_FRAMES_BUNDLE pFramesBundle  // Frames bundle received at SubmitFramesBundle
    );

// VTIO
// Implemented by the CIO2 driver.
// AVStream driver calls this function when it is called by the MF to uncache MDLs at its cancellation routine, in the case the cached buffer belongs
// to an SoC sensor and the buffer was given to the CIO2 driver and the CIO2 driver called CacheFramesBundle for this buffer. Calling this function
// actually requires the CIO2 to return the frames bundle to the AVStream, as the buffers of the bundle would be soon unpinned by the upper stack (the
// purpose of the cancellation routine is usually when the OS runs low of physical memory).
// It is not likely that the AVStream driver will call this API for RAW sensors when the frames bundle holds AVStream allocated ping pong capture buffers
// that are circulated between the AVStream and the CIO2 driver.
//
// If - the CIO2 driver can remove the frames bundle immediately from its device queues (FBPT) (e.g. the frames bundle buffers are not used by the CIO2
// device), then the CIO2 should
//      Remove its MDL pointers from the FBPT, remove its CachedBundleContext and the MDLs from its driver cache, call the
//      AVStream's ReleaseCacheFramesBundle and only then call the FramesBundleComplete callback of the AVStream driver.
//      Once the above is done, the API returns with a SUCCESS status (this is a synchronous call and response).
//      Note: This option seems challenging and hard to implement as the CIO2 driver is assumed to handle race conditions with the CIO2 device if it
//            wants to remove buffers that are already queued in the device FBPT.
// Else - the CIO2 driver cannot remove the frames bundle immediately (frames bundle buffers are queued in FBPT and the device may already access them)
//      The CIO2 driver is expected to gracefully uncache the frames bundle. This means that the CIO2 driver may wait for the opportunity
//      that this frame bundle buffers are filled by the device and have to be returned to the AVStream driver anyway.
//      In such case the CIO2 driver responds with CANCEL_PENDING status (this is a asynchronous call and response).
//      Once the frame bundle buffers are filled (folloing interrupt), before the frame bundle is returned to the AVStream, the CIO2 driver should remove
//      its MDL pointers from the FBPT, remove its CachedBundleContext and the MDLs from its driver cache, call the AVStream's ReleaseCacheFramesBundle and
//      only then call the FramesBundleComplete callback of the AVStream driver. This flow guarantees that the AVStream would be able to perform
//      its cancellation only after the CIO2 driver assures that the CIO2 device would not access the frames bundle's buffers.
//      Note: When the returned status is CANCEL_PENDING, the AVStream shall not call this function again for the same frames bundle!
//
// Returned ststus:
// SUCCESS - If CIO2 cancelled the cached frames bundle immediately (the frames bundle is not currently held by the CIO2 driver, but rather was handed
//          back to the AVStream, so that the CIO2 driver just has to delete it from the driver caching.
// CANCEL_PENDING - Cache cancellation cannot happen immediately (possibly because the frames bundle buffers are queued or are currently filled by the
//      device). The AVStream should wait for the CIO2 driver to call its ReleaseCacheFramesBundle callback. This will be the AVStream's acknowledge that
//      the command has been executed.
// FAIL_NO_SUCH_HANDLE - in case the specified frames bundle handle is not cached by the CIO2.
typedef NTSTATUS(*CANCEL_CACHE_FRAMES_BUNDLE)(
    PDATA_STREAM_INTERFACE DataStreamInterface,
    PVOID StreamContext,                // The context for the specific stream. Provided by the CSI2 host controller driver at the end of CreateSensorDataLink
    PIMAGE_FRAMES_BUNDLE pFramesBundle  // Frames bundle that its caching at the CIO2 has to be cancelled. Note: The CachedBundleContext in the pFramesBundle
                                        // identifies uniquely which bundle caching has to be cancelled.
    );

// VTIO
// SkyCam AVStream driver should query the CSI2 host controller driver to
// get the MIPI links security attributes. The CIO2 driver can get this 
// information from its CIO2 trustlet.
//
// Returned ststus:
// SUCCESS - If CIO2 detects its trustlet. The MipiLinksSecurityAssignments is meaningful
// Other ststus - error The platform does not support VTIO.
typedef NTSTATUS(*QUERY_MIPI_LINKS_SECURITY_ASSIGNMENT)(
    PMIPI_LINKS_SECURITY_ASSIGNMENT *MipiLinksSecurityAssignments
    );


// VTIO
// Enumeration of MIPI_LinkSecureState
typedef enum
{
    LINK_NC = 0,            // MIPI link is not connected.
    LINK_NON_SECURED = 1,   // MIPI link is set to operate in non secure mode
    LINK_SECURED = 2,       // MIPI link is set to operate in secure mode
} MIPI_LINK_SECURITY_STATE;

// VTIO
// SkyCam AVStream driver may query the MIPI link's current security state. The CIO2 driver must get this 
// information from its CIO2 trustlet.
//
// Returned ststus:
// SUCCESS - security state provided
// FAIL_SM_VTIO_NOT_AVAILABLE - OS does not support VTIO
// FAIL_SM_TRUSTLET_NOT_AVAILABLE - trustlet does not respond /not available
// FAIL_SM_STATE_BAD_LINK - This error code is returned when attempt to quesry link's security state for a link that is not supported on this
//                          platform (per the BIOS Manifest to VTL1 CIO2 driver)
typedef NTSTATUS(*QUERY_MIPI_LINK_SECURITY_STATE)(
    UCHAR LinkUsed,                           // Link used by this sensor stream. The value for each of the platform sensors depends on platform routing.
    // Valid numbers are:
    //      0 - This sensor uses link0. A high resolution sensor that uses two links must put in this field a value
    //          of 0 and LanesUsed is 8.
    //      1 - This sensor uses link1
    //      2 - This sensor uses link2
    //      3 - This sensor uses link3
    // Values other than the listed are not supported.
    // The Sensor driver receives from the BIOS as part of its ACPI resources the LinkUsed parameter. The sensor
    // driver reports to the AVStream driver the LinkUsed and the AVStream driver reports it to the CSI2 HC driver.
    unsigned short *MipiLinkSecurityState           // One of MIPI_LINK_SECURITY_STATE
);

// VTIO
// SkyCam AVStream driver may set the MIPI link's security state. This API is useful only for dynamic
// enabled MIPI links. The CIO2 driver must get this information from its CIO2 trustlet.
//
// Returned ststus:
// SUCCESS - If CIO2 could switch the specified MIPI link to the requested security state
// FAIL_SM_VTIO_NOT_AVAILABLE - OS does not support VTIO
// FAIL_SM_TRUSTLET_NOT_AVAILABLE - trustlet does not respond /not available
// FAIL_SM_STATE_BAD_LINK - This error code is returned when attempt to change MIPI link's security state for a link that is not supported on this
//                          platform (per the BIOS Manifest to VTL1 CIO2 driver)
// FAIL_SM_STATE_NOT_SUPPORTED - if the requested secure state is not supported
// FAIL_SM_STATE_CHANGE_NA - This error code is returned when attempt to change MIPI link's security state while its MIPI_LINK_SECURITY_ATTRIBUTE is
//                           not one of DYNAMIC_DEFAULT_NON_SECURED or DYNAMIC_DEFAULT_NON_SECURED
// FAIL_SM_STATE_CHANGE_WHILE_ACTIVE - This error code is returned when attempt to change MIPI link state while the link is active.
typedef NTSTATUS(*SET_MIPI_LINK_SECURITY_STATE)(
    PDATA_STREAM_INTERFACE DataStreamInterface,
    UCHAR LinkUsed,                          // Link used by this sensor stream. The value for each of the platform sensors depends on platform routing.
    // Valid numbers are:
    //      0 - This sensor uses link0. A high resolution sensor that uses two links must put in this field a value
    //          of 0 and LanesUsed is 8.
    //      1 - This sensor uses link1
    //      2 - This sensor uses link2
    //      3 - This sensor uses link3
    // Values other than the listed are not supported.
    // The Sensor driver receives from the BIOS as part of its ACPI resources the LinkUsed parameter. The sensor driver reports to the AVStream driver the
    // LinkUsed and the AVStream driver reports it to the CSI2 HC driver.
    unsigned short MipiLinkSecurityState           // One of MIPI_LINK_SECURITY_STATE
);

// VTIO
// Enumeration of DeviceSecureMode
typedef enum
{
    DEV_NON_SECURE_MODE = 0, // CIO2 operates in non secure mode (all links must be of non secure streams). Note: All secure streams must be stopped
                             // before moving CIO2 into this mode
    DEV_SECURE_MODE     = 1, // CIO2 operates in secure mode (all links must be of secure streams). Note: All non secure streams must be stopped before
                             // moving CIO2 into this mode.
    DEV_CONCURRENT_MODE = 2, // CIO2 operates in concurrent mode (supports secure and non secure streams concurrently). Note: This mode supports secure
                             // and non secure streams concurrently.
                             // This mode is not supported in KBL and BXT. This mode is currently reserved for CNL.
    DEV_VTIO_DISABLED_MODE=3 // CIO2 operates in VTIO disabled mode
} VTIO_DEVICE_SECURE_MODE;

// VTIO
// SkyCam AVStream driver may call this API to check the CIO2 trustlet secure mode.
// Note: - CIO2 driver start default mode is non secure, as backwards compatibility measure.
//
// Returned ststus:
// SUCCESS - If CIO2 could detect the trustlet's security mode
// FAIL_SM_VTIO_NOT_AVAILABLE - OS does not support VTIO
// FAIL_SM_TRUSTLET_NOT_AVAILABLE - trustlet does not respond /not available
typedef NTSTATUS(*QUERY_DEVICE_SECURE_MODE)(
    unsigned short *DeviceSecureMode        // One of VTIO_DEVICE_SECURE_MODE
    );

// VTIO
// SkyCam AVStream driver must call this API to switch the CIO2 driver and its device from its non secure
// default mode to secure mode, and at the end of secure session it must call this API to switch it back to non secure mode.
// AVStream driver must call this API before it calls CreateSensorDataLink for the 1st time in a new secure mode.
// Note: - CIO2 driver start default mode is non secure, as backwards compatibility measure.
//       - When a successful switch to SECURE_MODE, CIO2 resources are protected by its trustlet and become unavailble to VTL0 driver
//       - When a successful switch to NON_SECURE_MODE, CIO2 resources are un protected by its trustlet and become available to VTL0 driver, with the exception
//         of the MIPI ports control registers which are always kept secured in order not to allow malicious VTL0 driver to capture from a secure MIPI port.
//
// Returned ststus:
// SUCCESS - If CIO2 could switch to the requested security mode
// FAIL_SM_VTIO_NOT_AVAILABLE - OS does not support VTIO
// FAIL_SM_TRUSTLET_NOT_AVAILABLE - trustlet does not respond /not available
// FAIL_SM_MODE_NOT_SUPPORTED - if the requested secure mode is not supported
// FAIL_SM_NON_SECURE_STREAMS_ACTIVE - This error code is returned when attempt to move to secure mode while non secure streams are still active
// FAIL_SM_SECURE_STREAMS_ACTIVE - This error code is returned when attempt to move to non secure mode while secure streams are still active
// FAIL_SM_PROTECTION_ERROR - This error code is returned when attempt to move to secure mode and failed to protect CIO2 resources
// FAIL_SM_UNPROTECTION_ERROR - This error code is returned when attempt to move to non secure mode and failed to un protect CIO2 resources
typedef NTSTATUS(*SET_DEVICE_SECURE_MODE)(
    PDATA_STREAM_INTERFACE DataStreamInterface,
    unsigned short DeviceSecureMode         // One of VTIO_DEVICE_SECURE_MODE
    );
#endif

// Set Flash LED
// This API associates the flash LED to a specific stream and arms it to operate starting from the near frame start of this stream
// Tyically the flash LED services one sensor, and it does not make sense to activete it for different sensors. It is not the responsibility of the CIO2 driver to make sure
// the association of the flash LED to a specific sensor makes sense or not.
// Returned STATUS may be either:
//		- SUCCESS: Flash LED was armed successfuly
//		- BUSY: In case the FLASH LED is already active and is waiting for a next frame trigger
//		- BAD_STREAM: In case the StreamContext is not correct (AVStream did not create a link for that stream yet)
typedef NTSTATUS(*DATASTR_TRIGGER_FLASH)(
    PDATA_STREAM_INTERFACE DataStreamInterface,
    PVOID StreamContext,		// The context for the specific stream. Provided by the CSI2 host controller driver at the end of CreateSensorDataLink

    FLASH_TRIGGER_PARAMETERS *FlashTriggerParameters // These parameters come from the Sensor driver by its SdArmFlash API. See PMIC programming C and H files for details.
    );

typedef NTSTATUS(*DATASTR_RUN)(
    PDATA_STREAM_INTERFACE DataStreamInterface,
    PVOID StreamContext		// The context for the specific stream. Provided by the CSI2 host controller driver at the end of CreateSensorDataLink
    );

typedef NTSTATUS(*DATASTR_STOP)(
    PDATA_STREAM_INTERFACE DataStreamInterface,
    PVOID StreamContext		// The context for the specific stream. Provided by the CSI2 host controller driver at the end of CreateSensorDataLink
    );

// This is a query API for the instance ID of the driver/HW pair. It is useful in the case that there are more than one instance of a host controller. It
// is typically used before starting a stream.
typedef NTSTATUS(*QUERY_HOST_CONTROLLER_INSTANCE_ID)(
    IN PDATA_STREAM_INTERFACE DataStreamInterface,
    OUT PHOST_CONTROLLER HostController // Driver fills this output structure per its instance 
    );

// In case the MipiDataType is not supported by the CSI2 host controller, then the returned status would be NOT_SUPPORTED_FORMAT. Otherwise, the status 
// would be SUCCESS and the output CaptureDataFormat would be filled with the supported output formats for the required input MIPI data type.
// This is a query API for the capabilities of the host controller and is not stream specific. It is typically used before starting a stream and hence it
// does not require a StreamContext.
typedef NTSTATUS(*MIPI_DT_TO_CAPTURE_DATA_FORMAT)(
    IN PDATA_STREAM_INTERFACE DataStreamInterface,
    IN MIPI_DATA_TYPE MipiDataType,	// Data Type used for the sensor's transmission on this VC
    PSUPPORTED_CAPTURE_DATA_FORMATS CaptureDataFormats // Output data formats that are supported by the CSI2 host controller for the required Mipi Data
    // Type.
    // This function should respond per the following table (for SRP CSI2 host controller implementation):
    //
    // SupportedCaptureDataFormats	NumOfSupportedDataFormats	MipiDataType
    // ---------------------------	-------------------------	-----------------
    //	RAW8						1							RAW8p
    //	RAW10						1							RAW10p
    //	RAW12						1							RAW12p
    //	RAW14						1							RAW14p
    //	YUV422_8b					2							YUY2_LEp64
    //															UYVY_LEp64
    //	YUV420_8b_CSPS				2							NV12_LEp64
    //															NV21_LEp64
    //	RGB888						2							RGBA_LEp64
    //															BGRA_LEp64
    //	RGB565						2							RGB565_3ppc_LEp64
    //															RGB565_4ppc_LEp64
    //	EMBEDDED_8					1							PRI_EMBEDDED8p
    //	GSP_CODE_1					1							GSP_EMBEDDED8p
    //	GSP_CODE_2					1							GSP_EMBEDDED8p
    //	GSP_CODE_3					1							GSP_EMBEDDED8p
    //	GSP_CODE_4					1							GSP_EMBEDDED8p
    //	GSP_CODE_5					1							GSP_EMBEDDED8p
    //	GSP_CODE_6					1							GSP_EMBEDDED8p
    //	GSP_CODE_7					1							GSP_EMBEDDED8p
    //	GSP_CODE_8					1							GSP_EMBEDDED8p
    );

// In case the CaptureDataFormat is not supported by the CSI2 host controller, then the returned status would be NOT_SUPPORTED_MIPI_DATA_TYPE. Otherwise,
// the status would be SUCCESS and the output MipiDataTypes would be filled with the supported MIPI data types for the required output Capture data format.
// This is a query API for the capabilities of the host controller and is not stream specific. It is typically used before starting a stream and hence it 
// does not require a StreamContext.
typedef NTSTATUS(*CAPTURE_DATA_FORMAT_TO_MIPI_DT)(
    IN PDATA_STREAM_INTERFACE DataStreamInterface,
    IN CAPTURE_DATA_TYPE CaptureDataFormat,			// Data Type used for the sensor's transmission on this VC
    PSUPPORTED_MIPI_DATA_TYPES MipiDataTypes			// Output data formats that are supported by the CSI2 host controller for the required Mipi Data
    // Type.
    // This function should respond per the following table (for SRP CSI2 host controller implementation):
    //
    // CaptureDataFormat	NumOfSupportedDataTypes	SupportedMipiDataTypes
    // -----------------	-----------------------	----------------------
    //	RAW8p				1						RAW8
    //	RAW10p				1						RAW10
    //	RAW12p				1						RAW12
    //	RAW14p				1						RAW14
    //	YUY2_LEp64			1						YUV422_8b
    //	UYVY_LEp64			1						YUV422_8b
    //	NV12_LEp64			1						YUV420_8b_CSPS
    //	NV21_LEp64			1						YUV420_8b_CSPS
    //	RGBA_LEp64			1						RGB888
    //	BGRA_LEp64			1						RGB888
    //	RGB565_3ppc_LEp64	1						RGB565
    //	RGB565_4ppc_LEp64	1						RGB565
    //	PRI_EMBEDDED8p		1						EMBEDDED_8
    //	GSP_EMBEDDED8p		8						GSP_CODE_1
    //												GSP_CODE_2
    //												GSP_CODE_3
    //												GSP_CODE_4
    //												GSP_CODE_5
    //												GSP_CODE_6
    //												GSP_CODE_7
    //												GSP_CODE_8
    );

typedef NTSTATUS(*DATASTR_SUBMIT_FRAMES_BUNDLE)(
    // If the VC_Value in pFrame->VC_Value is not one of those defined in CreateSensorDataLink at the LinkConfig.VcSettings.VC_Value,an error code will be returned.
    // In a sensor that makes use of more than one VC, SubmitFramesBundle has to be called for each of its VCs.
    PDATA_STREAM_INTERFACE DataStreamInterface,
    PVOID StreamContext,	// The context for the specific stream. Provided by the CSI2 host controller driver at the end of CreateSensorDataLink
    PIMAGE_FRAMES_BUNDLE pFramesBundle
    );

// This API is useful only when the frame report mode is FRAME_REPORT_REACHED_FRAME_LINE
typedef NTSTATUS(*DATASTR_COMPLETED_FRAMES_BUNDLE_PROCESSING)(
    PDATA_STREAM_INTERFACE DataStreamInterface,
    PVOID StreamContext,	// The context for the specific stream. Provided by the CSI2 host controller driver at the end of CreateSensorDataLink
    PIMAGE_FRAMES_BUNDLE pFramesBundle	// This is the frames bundle that the CSI2 host controller driver reported the partial frame buffer to the AVStream driver
    // with frame status of FRAME_REPORT_REACHED_FRAME_LINE. 
    );


// Get link statistics - The AVStream driver can get through this API the statistics that the CSI2 HC driver collected for a specific link. Statistics mostly include all kind of
// errors that are detected outside a frame boundary (e.g. frame end without frame start, packet header ECC error, etc., TBD). Per frame errors are typically reported in part of
// FramesBundleComplete status (part of FramesBundleComplete). To simplify the use of debug and tuning applications, the GetStatistics response (pStatistics) will include counters for frame
// errors (that are reported in FramesBundleComplete as well).
// Currently there is no indication to the AVStream driver that the statistics counters have been updated (neither a threshold on number of such updates). It seems that these statistics
// are of interest mostly for platform debug and build verification as most of these statistics cannot result with SW actions (TBD). The current assumption is that the AVStream driver
// would call this API when a debug tool asks the AVStream to provide such information.
// When a stream is established (at CreateSensorDataLink), all stream related statistics counters are zeroed.
typedef NTSTATUS(*DATASTR_GET_LINK_STATISTICS)(
    PDATA_STREAM_INTERFACE DataStreamInterface,
    PVOID StreamContext,	// The context for the specific stream. Provided by the CSI2 host controller driver at the end of CreateSensorDataLink

    PLINK_STATISTICS pStatistics
    );

// The following function serves the AVStream driver when it creates the presentation time for every captured frame.
// AVStream resolves from the captured buffer the DescriptorTimeStamp (IMAGE_FRAMES_BUNDLE.TimeStamp). Then it performs the following:
//		KSHEADER.PresentationTime = IKsReferenceClock::GetCorrelatedTime +
//									(ULONGLONG)DescriptorTimeStamp - GetCurrentCIO2TimeStemp();
typedef ULONGLONG(*DATASTR_GET_CURRENT_TIMESTAMP)(
    PDATA_STREAM_INTERFACE DataStreamInterface
    // PVOID StreamContext,	// No need for strem context as the timestamp is a global timer at the CIO2 controller, that ticks for all the sensors

    // This function returns the following value:
    //		return ((ULONGLONG)ReadReg32(Csi2TimeStampReg) + gTimeStampScroll);
    );

/*******************
* \brief Typedef for DataStreamInterface init function
* \todo Add a sensor context, not only the interface
*
* @param DataStreamInterface The interface pointer
*
* @param AVStreamDriverContext The AVStream driver's context. This context will be provided by the interface implmenter (IO driver) with every callback
*
* @param PortConfig Identifies the sensor within the implementer. Using this structure the implementer is able to determine the sensor connectivity and
* identify the sensor
*
* @param DataFormat A definition of the data format given by the sensor
*
* @param DataRate The data rate expected. Measured in milliseconds from frame N start to frame N+1 start (single cycle)
*
* @param ReportMode Frame reporting mode (e.g. end-of-frame, start-of-next-frame)
*
* @param FramesBundleComplete A callback to be provided by the AVStream driver. This callback will be called for each frame received or buffer returned empty.
*
*******************/
typedef NTSTATUS(*DATASTR_CREATE_SENSOR_DATA_LINK)(
    IN PDATA_STREAM_INTERFACE DataStreamInterface,
    IN PVOID AVStreamDriverContext,			// CSI2 host controller driver should save this context for later call into the AVStream callback functions with
    // every frame from this stream

    IN PSENSOR_LINK LinkConfig,				// Link configuration details for the sensor in this host controller. The sensor driver would use same context
    // for commands for this sensor. AVStream Driver will get this configuration details from the sensor driver when
    // they establish their communication.
    // Link configuration will typically come as ACPI resources to the Sensor Driver. The Sensor driver makes use of
    // some of the link configuration parameter such as:
    //		- Number of links this sensor use. Typical sensors consume a single link, however some high resolution
    //		  sensors use two links (our CSI2 host controller requires in this case that the 1st link is link 0 and
    //		  the 2nd link is link1)
    //		- Link used by this sensor stream. Valid numbers are 0 or 1 or 2 or 3. A high resolution sensor that
    //		  uses two links must put in this field a value of 0 and its number of links is 2. 
    //		- Number of lanes that this link consumes 
    
    IN USHORT EstimatedNumberOfFrames,		// AVStream driver should provide at this parameter its estimation form back to back number of frames that it will
    // submit to the CIO2 driver at the SubmitFramesBundle API. This would allow the CIO2 driver to create an FBPT list of a
    // proper depth of entries (when each entry represents a frame buffer). In case of RAW sensor the AVStream driver allocates the input
    // capture buffers and can provide an accurate depth size to allow the CIO2 driver to hold all buffers in FBPT entries. In case of SoC sensor
    // the AVStream driver does not allocate the capture buffers and can provide a guess (that is based on its exposed filter PIN parameters
    // definition).
    // A value of 0 means that the AVStream driver cannot provide a guess. In such case the CIO2 driver would allocate a 64 number of FBPT entries
    // and in case there are more submitted frames than that depth, the CIO2 driver would queue the rest buffers in a SW queue until an FBPT entry
    // is freed as a result of frame end detection.
    // A non 0 value means that the AVStream driver knows how many buffers it will submit to the CIO2 driver (submission of multiple buffers allows
    // the CIO2 to capture multiple frames without interrupts to the driver). In such case the CIO2 driver would allocate the specified number of
    // FBPT entries and in case there are more submitted frames than that depth, the CIO2 driver would queue the rest buffers in a SW queue until
    // an FBPT entry is freed as a result of frame end detection.

    IN FRAMES_BUNDLE_REPORT_MODE ReportMode,
    IN SHORT ThresholdLineNumber,			// This parameter is meaningful only when ReportMode is FBRM_FRAMES_BUNDLE_LINE_NUMBER. When used, the CSI2 host controller driver puts this value in the
    // LineNumINT field in CDMAC1 register. When used, this number must be between 1 and 0x3FFF.
    IN SHORT LineNumberUpdate,				// This parameter is meaningful only when ReportMode is FBRM_FRAMES_BUNDLE_LINE_NUMBER. When used, the CSI2 host controller driver puts this value in the
    // LineNumberUpdate field in CDMAC1 register. When used, this number must be between 1 and 0x1FF. This parameter tells the host controller how
    // often should it update the LineNumber at the FBPT descriptor in host memory. In this mode of operation the ISP is interested in chunks of lines
    // hence the LineNumber should be updated at this rate (faster rate is not useful for the ISP, but is a possible and valid vaule). 

    // Implemented by the AVStream driver. CSI2 host controller driver calls this callback function when a frame has been completed by the IO driver. This
    // will only be called if the NotifyOnCompletion indication is set in the IMAGE_FRAMES_BUNDLE for the frames bundle.
    IN FRAMES_BUNDLE_COMPLETE FramesBundleComplete,

#ifdef SECURE_INPUT_FRAME
    // VTIO
    // Implemented by the AVStream driver. CSI2 host controller driver calls this callback function when it receives a frame bundle that is not yet cached (e.g. the
    // CachedBundleContext at the IMAGE_FRAMES_BUNDLE is NULL).
    IN CACHE_FRAMES_BUNDLE CacheFramesBundle,

    // VTIO
    // Implemented by the AVStream driver. CSI2 host controller driver calls this callback function when it releases a frame bundle that is already cached (e.g. the
    // CachedBundleContext at the IMAGE_FRAMES_BUNDLE is not NULL).
    IN RELEASE_CACHE_FRAMES_BUNDLE ReleaseCacheFramesBundle,
#endif

    OUT PVOID *StreamContext				// A handle returned by the CSI2 host controller driver. It identifies the created stream context.
    // This is the AVStream driver's stream context within the CSI2 host controller driver. It keeps inside, 
    // information like the LINK_ID for the created sensor, sensor data queue details, etc.
    // AVStream driver receives this output value (a handle for further calls) at return from Create Sensor Data
    // Link, and should use it with every further command to the CSI2 host controller driver (SubmitFramesBundle,
    // GetStatistics, Run, Stop).

    // Comment INT_ON_LINE_NUM
    // OUT Pointer to the FBPT of the data buffer that is related to Interrupt on line number of this stream. This would allow the ISP FW to look at the CurLineNumb to make sure that the IO
    //								device did a meaningful progress 

    // Note: STATUS that is returned by this function may be:
    //		SUCCESS
    //		DATASTR_LINK_IS_ALREADY_IN_USE	- in case the link has been created and not yet stopped
    //		TBD - more errors
    );

/*******************
* \brief Typedef for DataStreamInterface init function
* \todo Add a sensor context, not only the interface
*
* @param InterfaceHeader A WDF defined structure that provides the basic interface support, such as reference count.
*
* @param CreateSensorDataLink  Method provided by the Host Controller driver. The AVStream driver calls this to establish a stream to a single sensor
*						connection.
*						All of the information needed to perform the initial setup and configuration of the DMA engines, etc., is provided here, including
*						the sensor to connect to, the data Link of the sensor, and the expected data rate and data format.
*						Once this method returns the data interface is ready for use and SubmitFramesBundle() & Run() may be called.
*						Note that SetCapture() must be proceeded by a Stop() command.
*
* @param SubmitFramesBundle When the AVStream has an image buffer ready to be filled, it is provided to the CSI3 Host Controller driver using this method. The MDLs
*						provided in the IMAGE_FRAMES_BUNDLE structure are pinned by the AVStream driver, ready to be written into the FrameBufferPointer
*						structures that the DMA engines use. The AVStream should provide at least one, and typically two or three buffers in advance of the
*						current frame being filled to ensure continuous service.
*						The AVStream driver must make sure to provide enough empty frames to the data interface
*
* @param Run AVStream driver commands the Host Controller driver to start execution for the specified channel. After this command, the host controller
*						will start its DMA engines to read host descriptors and be ready to write to the pointed output buffers once the specified sensor
*						streams data. The Host Controller will use frames given in SubmitFramesBundle(). If no frames are provided - the Host Controller must drop
*						the frames
*
* @param Stop AVStream driver commands the Host Controller driver to stop execution for the specified channel. While this command executes, the CSI3 host
*						controller driver halts its DMA engines and then it calls the CompleteFrame API in the AVStream driver to return every buffer that
*						it received with a HOST_CONTEROLLER_HALTED or FRAME_EMPTY (?) completion status.
*						At the completion of this command, the host controller DMA engines for the specified channel are halted and the CSI3 host controller
*						driver returns with a SUCCESS status (all buffers already returned to the AVStream driver).
*/
typedef struct _DATA_STREAM_INTERFACE
{
    struct _INTERFACE InterfaceHeader;

    //TODO--->GET_CAPABILITIES GetCapabilities ... such as flash trigger TRUE / FALSE, more ?
    //TODO--->Make sure we have interfac version(I think there is one in the INTERFACE) to allow extensions and awareness of the calling
    //driver.Should be done in other driver interfaces as well.

#ifdef SECURE_INPUT_FRAME
    // VTIO
    QUERY_MIPI_LINKS_SECURITY_ASSIGNMENT QueryMipiLinkSecurityAssignment;

    // VTIO
    QUERY_MIPI_LINK_SECURITY_STATE QueryMipiLinkSecurityState;

    // VTIO
    SET_MIPI_LINK_SECURITY_STATE SetMipiLinkSecurityState;

    // VTIO
    QUERY_DEVICE_SECURE_MODE QueryDeviceSecureMode;

    // VTIO
    SET_DEVICE_SECURE_MODE SetDeviceSecureMode;
#endif

    // There can be more than one CSI2 host controller. SkyCam AVStream driver
    // should query the CSI2 host controller ID before it makes use of any of the
    // private interfaces it detects.
    // Each of the defined host controllers (CSI2 or CSI3 or USB) must identify
    // itself and match a predefined GUID at its InterfaceHeader. When there is
    // more than one host controller HW instance of same HW type, then this
    // query function would allow the SkyCam AVStream driver to connect to the 
    // right one that matches the ControllerInstanceID that was provided by the
    // BIOS to the sensor driver for which the stream is about to be created.
    QUERY_HOST_CONTROLLER_INSTANCE_ID QueryHostControllerInstanceID;

    // SkyCam AVStream driver should query the CSI2 host controller driver before
    // it calls CreateSensorDataLink with the selected input MIPI data type to
    // get the possible output data formats for this MIPI datat type.
    // One of these output capture data formats should match the OS expected
    // capture buffer formats, or the ISP expected buffer formats.
    // This API is less useful than the QueryCaptureDataFormatToMipiDt.
    MIPI_DT_TO_CAPTURE_DATA_FORMAT QueryMipiDtToCaptureDataFormat;

    // SkyCam AVStream driver should query the CSI2 host controller driver before
    // it calls CreateSensorDataLink with the required output CaptureDataFormat to
    // get the possible MIPI data types.
    // One of these data types should be set at the sensor by the sensor driver.
    // This API is the preferred one (QueryMipiDtToCaptureDataFormat is less useful).
    CAPTURE_DATA_FORMAT_TO_MIPI_DT QueryCaptureDataFormatToMipiDt;

    // Set sensor link parameters (number of links used by this sensor, number of
    // lanes that the sensor uses, data format, compression, frame rate, number 
    // of VCs, etc.
    //
    // This API returns a StreamContext. The CSI2 host controller supports up to 4
    // links on a single interface.
    DATASTR_CREATE_SENSOR_DATA_LINK CreateSensorDataLink;

    // Provide a buffer to be added to the queue. AVStream driver submits capture
    // buffers through this API. 
    // Note: This function may be called from within the FramesBundleComplete callback

    DATASTR_SUBMIT_FRAMES_BUNDLE SubmitFramesBundle;

    // This API is reserved for the usage model of FRAMES_BUNDLE_REPORT_MODE of
    // FBRM_FRAMES_BUNDLE_LINE_NUMBER (future option). In this case the CSI2 HC driver will
    // report the AVStream driver that the current frame buffer fill reached the
    // required line number. At the end of ISP processing of the frame, the
    // AVStreram driver will call this API to notify the CSI2 host controller driver
    // that the ISP is done with this frame descriptor and its buffers and the FBPT
    // descriptor that was used by the ISP to make sure the current captured line
    // number already passed the next ISP processing space to make sure there are
    // no underruns.
    // As a result, the CSI2 host controller driver shall verify that the CSI2 host
    // controller HW already completed this frame (otherwise it should return an
    // error that means that buffer synchronization failed - this case is fatal SW
    // error). In the normal case, the CSI2 host controller driver shall dequeue
    // this frame's descriptors from its HW (like it does when interrupt on completion
    // is detected) and calls the FramesBundleComplete API - now with
    // FRAME_REPORT_END_OF_FRAME status. Note that the CSI2 HC driver will call the
    // FramesBundleComplete API within the context of CompletedFramesBundleProcessing.
    // Note: This function may be called from within the FramesBundleComplete callback.
    DATASTR_COMPLETED_FRAMES_BUNDLE_PROCESSING CompletedFramesBundleProcessing;

#ifdef SECURE_INPUT_FRAME
    // VTIO
    CANCEL_CACHE_FRAMES_BUNDLE CancelCacheFramesBundle;
#endif

    // Get link statistics - The AVStream driver can get through this API the
    // statistics that the CSI2 HC driver collected for a specific link. Statistics
    // mostly include all kind of errors that are detected outside a frame boundary
    // (e.g. frame end without frame start, packet header ECC error, etc., TBD).
    DATASTR_GET_LINK_STATISTICS GetLinkStatistics;

    // Get GetCurrentIoTimeStemp
    DATASTR_GET_CURRENT_TIMESTAMP GetCurrentIoTimeStamp;

    // Set Flash LED trigger
    // This API associates the flash LED to a specific stream and arms it to operate
    // starting from the near frame start of this stream.
    DATASTR_TRIGGER_FLASH TriggerFlash;

    // Run the specified link stream
    // Running a link's stream can be called only after the SkyCam AVStream driver
    // called successfuly the CreateSensorDataLink API and preferably submitted
    // couple of capture frame buffers by calling the SubmitFramesBundle API.
    DATASTR_RUN Run;

    // Stop the specified link stream
    // After stopping a stream, the SkyCam AVStream driver must call
    // CreateSensorDataLink and SubmitFramesBundle again before it calls Run. Once all
    // created links are stopped, the CSI2 host controller driver would power off
    // (D0i3) its core.
    // The SkyCam AVStream may choose to dereference the DATA_STREAM_INTERFACE
    // after it calls Stop for all the running streams. 
    DATASTR_STOP Stop;

} DATA_STREAM_INTERFACE, *PDATA_STREAM_INTERFACE;

static const GUID GUID_SENSOR_DATA_INTERFACE =
{ 0xf08b1823, 0x5b45, 0x4cc9, { 0xbd, 0x78, 0x37, 0xab, 0xc4, 0x59, 0x76, 0x16 } };

#ifdef SECURE_INPUT_FRAME
#define GUID_SENSOR_DATA_INTERFACE_VERSION     3
#else
#define GUID_SENSOR_DATA_INTERFACE_VERSION     2
#endif

#endif
