/*************************************************************************************************
// INTEL CONFIDENTIAL Copyright 2011-2018 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to the source code
// ("Material") are owned by Intel Corporation or its suppliers or licensors. Title to the Material
// remains with Intel Corporation or its suppliers and licensors. The Material contains trade secrets
// and proprietary and confidential information of Intel or its suppliers and licensors. The Material is
// protected by worldwide copyright and trade secret laws and treaty provisions. No part of the
// Material may be used, copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual property right is
// granted to or conferred upon you by disclosure or delivery of the Materials, either expressly, by
// implication, inducement, estoppel or otherwise. Any license under such intellectual property
// rights must be express and approved by Intel in writing.
//*************************************************************************************************/

#ifndef SENSOR_TYPES_H
#define SENSOR_TYPES_H

#include "platf_info.h"
#include <minwindef.h>

/*!
 * \brief Resolution ratio
 */
#define RATIO_MUX 1000                              /*!< Resolution Ratio Mux */
#define GET_RATIO(a, b) ((a) * RATIO_MUX / (b))     /*!< Resolution Ratio Converter */

#define ASPECT_RATIO_IS_EQUAL(a, b)             ((abs((a) - (b))) < (RATIO_MUX / 20))
#define ASPECT_RATIO_IS_EQUAL_16to9(a)          ASPECT_RATIO_IS_EQUAL(a, GET_RATIO(16, 9))
#define ASPECT_RATIO_IS_EQUAL_4to3(a)           ASPECT_RATIO_IS_EQUAL(a, GET_RATIO(4, 3))

/*!
 * \brief Panel Orientation of Sensor
 */
#define PANEL_LEFT  2                               /*!< Panel Left Camera, */
#define PANEL_RIGHT 3                              /*!< Panel Right Camera, */
#define PANEL_FRONT 4                               /*!< Panel Front Camera, User Face */
#define PANEL_BACK  5                               /*!< Panel Back Camera, World Face */

/*!
 * \brief Function Type of Sensor
 */
#define  TYPE_RAW                       0           /*!< Sensor used as a raw sensor */
#define  TYPE_YUV                       1           /*!< Sensor used as a yuv sensor */
#define  TYPE_EXT                       2           /*!< Sensor used as a external ISP*/
#define  TYPE_ET                        3           /*!< Sensor used as an eye track special sensor */
#define  TYPE_MOCK                      4           /*!< Sensor mock */
#define  TYPE_IVCAM_USB_RAW             5           /*!< IVCAM USB RAW sensor */
#define  TYPE_MONOIR                    6           /*!< Sensor used as a mono ir sensor*/
#define  TYPE_MONO                      7           /*!< Sensor used as a mono sensor*/
#define  TYPE_RAWIR                     8           /*!< Sensor used as a rawir sensor*/
#define  TYPE_RGBIR                     9           /*!< Sensor used as a RGB-IR sensor*/
#define  TYPE_TPG                       10          /*!< Sensor used as a mipi-gen, which is not the real sensor*/


/*!
 * \brief Sensor output format
 *
 * \Sensor output format is the same as ISP streaming input format. same as ShCss_input_format data enum
 */
#define INPUT_FORMAT_YUV420_8_LEGACY    0           /*!< 8 bits per subpixel */
#define INPUT_FORMAT_YUV420_8           1           /*!< 8 bits per subpixel */
#define INPUT_FORMAT_YUV420_10          2           /*!< 10 bits per subpixel */
#define INPUT_FORMAT_YUV422_8           3           /*!< UYVY..UVYV, 8 bits per subpixel */
#define INPUT_FORMAT_YUV422_10          4           /*!< UYVY..UVYV, 10 bits per subpixel */
#define INPUT_FORMAT_RGB_444            5           /*!< BGR..BGR, 4 bits per subpixel */
#define INPUT_FORMAT_RGB_555            6           /*!< BGR..BGR, 5 bits per subpixel */
#define INPUT_FORMAT_RGB_565            7           /*!< BGR..BGR, 5 bits B and $, 6 bits G */
#define INPUT_FORMAT_RGB_666            8           /*!< BGR..BGR, 6 bits per subpixel */
#define INPUT_FORMAT_RGB_888            9           /*!< BGR..BGR, 8 bits per subpixel */
#define INPUT_FORMAT_RAW_6              10          /*!< RAW data, 6 bits per pixel */
#define INPUT_FORMAT_RAW_7              11          /*!< RAW data, 7 bits per pixel */
#define INPUT_FORMAT_RAW_8              12          /*!< RAW data, 8 bits per pixel */
#define INPUT_FORMAT_RAW_10             13          /*!< RAW data, 10 bits per pixel */
#define INPUT_FORMAT_RAW_12             14          /*!< RAW data, 12 bits per pixel */
#define INPUT_FORMAT_RAW_14             15          /*!< RAW data, 14 bits per pixel */
#define INPUT_FORMAT_RAW_16             16          /*!< RAW data, 16 bits per pixel */
#define INPUT_FORMAT_BINARY_8           17          /*!< Binary byte stream. */

/*!
 * \Brief Sensor PDAF T2 RAW10 support
 *
 * \Describes if long MIPI packets have DT with some other DT format.
 */

typedef enum _ISYS_MIPI_DT_RENAME_MODE
{
        ISYS_MIPI_DT_NO_RENAME = 0,
        ISYS_MIPI_DT_RENAMED_MODE,
        N_ISYS_MIPI_DT_MODE
} ISYS_MIPI_DT_RENAME_MODE;

typedef enum _ISYS_MIPI_USERDEFINE_DATA_TYPE
{
        /* User defined 8-bit data type 1 */
        ISYS_MIPI_DATA_TYPE_USER_DEF1            = 0x30,
        /* User defined 8-bit data type 2 */
        ISYS_MIPI_DATA_TYPE_USER_DEF2            = 0x31,
        /* User defined 8-bit data type 3 */
        ISYS_MIPI_DATA_TYPE_USER_DEF3            = 0x32,
        /* User defined 8-bit data type 4 */
        ISYS_MIPI_DATA_TYPE_USER_DEF4            = 0x33,
        /* User defined 8-bit data type 5 */
        ISYS_MIPI_DATA_TYPE_USER_DEF5            = 0x34,
        /* User defined 8-bit data type 6 */
        ISYS_MIPI_DATA_TYPE_USER_DEF6            = 0x35,
        /* User defined 8-bit data type 7 */
        ISYS_MIPI_DATA_TYPE_USER_DEF7            = 0x36,
        /* User defined 8-bit data type 8 */
        ISYS_MIPI_DATA_TYPE_USER_DEF8            = 0x37,
        /** Reserved 0x38-0x3F */
} ISYS_MIPI_USERDEFINE_DATA_TYPE;


/*!
 * \Brief Sensor mode enum
 *
 * \different sensor mode will have different resolution and configuration support
 */
typedef enum _SNSR_MODE
{
    StillMode,                                      /*!< sensor still mode. */
    VideoMode,                                      /*!< sensor video mode. */
    PreviewMode,                                    /*!< sensor preview mode. */
    RawMode,                                        /*!< sensor raw mode. */
    maxMode                                         /*!< sensor mode number. */
} SNSR_MODE;

/*!
* \brief Sensor rotation enum
*
* \different sensor rotation (clockwise) will result in different support on pipeline and algorithm
*/
typedef enum _SENSOR_ROTATION
{
    Rotation_0   = 0x00,                                      /*!< rotate 0. */
    Rotation_45  = 0x01,                                      /*!< rotate 45. */
    Rotation_90  = 0x02,                                      /*!< rotate 90. */
    Rotation_135 = 0x03,                                      /*!< rotate 135. */
    Rotation_180 = 0x04,                                      /*!< rotate 180.*/
    Rotation_225 = 0x05,                                      /*!< rotate 225. */
    Rotation_270 = 0x06,                                      /*!< rotate 270. */ 
    Rotation_315 = 0x07,                                      /*!< rotate 315. */
} SENSOR_ROTATION;

/*!
 * \brief Sensor data input interface type enum
 */
typedef enum
{
    MIPI,                                           /*!< sensor using mipi as data input interface. */
    FIFO                                            /*!< sensor using fifo as data input interface, it is not a real sensor. */
} SNSR_INTERFACE_TYPE;

/*!
 * \brief Sensor data type enum
 */
typedef enum
{
    TPG,                                            /*!< sensor data is TPG, which is not a real sensor. */
    RAW,                                            /*!< sensor data is RAW. */
    SOC,                                            /*!< sensor data is SOC processed data. */
    EXTISP,                                         /*!< sensor data is External ISP processed data. */
    MONOIR,                                         /*!< sensor data is MONO IR. */
    MONO,                                           /*!< sensor data is MONO. */
    RGBIR                                           /*!< sensor data is RGB-IR. */
} SNSR_TYPE;

typedef enum
{
    NO_PDAF_SENSOR = 0,
    PDAF_SENSOR_T1,
    PDAF_SENSOR_T2,
    PDAF_SENSOR_T3,
    SUPPORT_PDAF_SENSOR_NUM
} PDAF_sensor_type;


/*!
 * \brief Shutter Type of Sensor
 */
typedef enum
{
    ROLLING_SHUTTER,                                /*!< Rolling shutter sensor. */
    GLOBAL_SHUTTER                                  /*!< Global shutter sensor. */
}SNSR_SHUTTER_TYPE;

/*!
 * \brief Sensor information data structure
 */
#define MAXSTRING_SENSOR    16                      /*!< Max charactor numbers of sensor string. */
#define MAXSTRING_MODULE    64                      /*!< Max charactor numbers of module string. */
#define MAXSTRING_INFO      64                      /*!< Max charactor numbers of information string. */
typedef struct _SNSR_INFO
{
    GUID guid;                                      /*!< Mandatory. Sensor device GUID. */
    char type;                                      /*!< Mandatory. Sensor output data type. */
    char format;                                    /*!< Mandatory. Sensor output data format. */
    char cmos[MAXSTRING_SENSOR];                    /*!< Mandatory. Name string of cmos. */
    char module[MAXSTRING_MODULE];                  /*!< Mandatory. Name string of module. */
    char info[MAXSTRING_INFO];                      /*!< Mandatory. Name string of platform information. */
    char infosub[MAXSTRING_SENSOR];                   /*!< Mandatory. Name string of platform sub information. */
    char silicon[MAXSTRING_SENSOR];                 /*!< Mandatory. Name string of silicon information. */
    char customer[MAXSTRING_SENSOR];                /*!< Mandatory. Name string of customer information. */
    char controllogic_id;
    ULONG mclk_port;
    WCHAR driverpath[MAX_PATH];
}SNSR_INFO, *PSNSR_INFO;

/*!
 * \brief Sensor verion information data structure
 */
typedef struct _SNSR_VERSION_INFO
{
    ULONG FwVersion;                               /*!< Optional. Sensor firmware version. */
    ULONG FwRevision;                              /*!< Optional. Sensor firmware revision. */
    ULONG SensorRevision;                          /*!< Optional. Sensor revision. */
    ULONG SensorId;                                /*!< Mandatory. Sensor ID. */
    ULONG ControllerVersion;                       /*!< Optional. Sensor controller version. */
    ULONG ControllerId;                            /*!< Optional. Sensor controller ID. */
    ULONG ControllerRevision;                      /*!< Optional. Sensor controller revision. */
    ULONG SystemClock;                             /*!< Optional. Sensor system clock */
} SNSR_VERSION_INFO;

/*!
 * \brief Sensor resolution information data structure
 */
typedef struct _SNSR_RESOLUTION
{
    ULONG Width;                                    /*!< Mandatory. Sensor resolution width. */
    ULONG Height;                                   /*!< Mandatory. Sensor resolution height. */
    ULONG CropX;                                    /*!< Optional. Sensor resolution crop in width. */
    ULONG CropY;                                    /*!< Optional. Sensor resolution crop in height. */
    LONG FPS;                                       /*!< Mandatory. Sensor resolution frame number per second. */
    ULONG BayerOrder;                               /*!< Mandatory. Sensor resolution bayer order. */
    ULONG BayerOrder180Degree;                      /*!< Mandatory. Sensor resolution bayer order when image mirror/flip. */
    ULONG IsFullFov;                                /*!< Mandatory. Whether the sensor resolution is full FOV. */
} SNSR_RESOLUTION;

/*!
* \brief PDAF data pattern, this should be aligned with ia_css_isys_type_paf
*/

typedef enum 
{
    NO_PAF,
    INTERLEAVED_PAF,
    /* PAF data in non-interleaved format(LL/RR or RR/LL) */
    NON_INTERLEAVED_PAF,
    /* Type 3 sensor types , PAF data embedded in Image Frame*/
    /* Frame Embedded PAF in interleaved format(RLRL or LRLR)*/
    FRAME_EMB_INTERLEAVED_PAF,
    /* Frame Embedded PAF non-interleaved format(LL/RR or RR/LL)*/
    FRAME_EMB_NON_INTERLEAVED_PAF,
    N_PATTERN_TYPE_PAF
} PAF_PATTERN_TYPE;

/*!
* \brief PDAF data bpp
*/
typedef enum
{
    PAF_INPUT_BPP_8_BITS = 0,
    PAF_INPUT_BPP_10_BITS
} PAF_INPUT_BPP;

/*!
* \brief RAW PDAF resolution information data structure, it only be used in PDAF sensor
*/
typedef struct _RAWPDAF_RESOLUTION
{
    ULONG sensorWidth;                              /*!< Mandatory. Sensor resolution width. */
    ULONG sensorHeight;                             /*!< Mandatory. Sensor resolution height. */
    ULONG PDAFWidth;                                /*!< Mandatory. PDAF resolution width. */
    ULONG PDAFHeight;                               /*!< Mandatory. PDAF resolution height. */
    ULONG fps;
} RAWPDAF_RESOLUTION;

/*!
* \brief PDAF resolution information data structure, it only be used in PDAF sensor
*/
typedef struct _PDAF_RESOLUTION
{
    ULONG PDAFWidth;                                /*!< Mandatory. PDAF resolution width. */
    ULONG PDAFHeight;                               /*!< Mandatory. PDAF resolution height. */
    ULONG fps;
} PDAF_RESOLUTION;

typedef struct _PDAF_RESOLUTION_SET
{
    ULONG Count;                                    /*!< Mandatory. Number of sensor PDAF resolutions. */
    PDAF_RESOLUTION* Resolutions;                   /*!< Mandatory. Pointer list of sensor PDAF resolutions. */
} PDAF_RESOLUTION_SET;
/*!
* \brief PDAF Caps information data structure, it only be used in PDAF sensor
*/
typedef struct _PDAF_CAPS
{
    BOOLEAN split_accross_multiple_lines;
    PAF_PATTERN_TYPE input_modes;
    ULONG input_bpp;
    RAWPDAF_RESOLUTION pdaf_resolution;                           /*!< Mandatory. PDAF resolution height. */
    ISYS_MIPI_DT_RENAME_MODE dt_rename_mode;
    ISYS_MIPI_USERDEFINE_DATA_TYPE data_type;
} PDAF_CAPS;

/*!
 * \brief Sensor resolution set data structure
 */
typedef struct _SNSR_RESOLUTION_SET
{
    ULONG Count;                                    /*!< Mandatory. Number of sensor resolutions. */
    SNSR_RESOLUTION* Resolutions;                   /*!< Mandatory. Pointer list of sensor resolutions. */
} SNSR_RESOLUTION_SET;

/*!
 * \brief Sensor capability data structure
 */
typedef struct _SNSR_CAPS
{
    SNSR_INTERFACE_TYPE InterfaceType;              /*!< Mandatory. Sensor interface type. */
    SNSR_TYPE SensorType;                           /*!< Mandatory. Sensor type. */
    ULONG AspectRatio;                              /*!< Optional. Sensor resolution aspect ratio. */
    ULONG MaxAperture;                              /*!< Optional. */
    ULONG MinAperture;                              /*!< Optional. */
    ULONG MaxExposureTime;                          /*!< Optional. */
    ULONG MinExposureTime;                          /*!< Optional. */
    ULONG MaxGain;                                  /*!< Optional. */
    ULONG MinGain;                                  /*!< Optional. */
    LONG  MaxFocusPos;                              /*!< Optional. */
    LONG  MinFocusPos;                              /*!< Optional. */
    LONG  DefaultFocusPos;                          /*!< Optional. */
    BOOLEAN FocusSupport;                           /*!< Mandatory. Focus support capability. */
    BOOLEAN FlashSupport;                           /*!< Mandatory. Flash support capability. */
    BOOLEAN PreciseScaling;                         /*!< Mandatory. Precise scaling support capability. */
    LONG FovFocalLengthX;                           /*!< Mandatory. Fov focal length in X. */
    LONG FovFocalLengthY;                           /*!< Mandatory. Fov focal length in Y. */
    LONG CamAngleOffsetPitch;                       /*!< Mandatory. Camera angle offset pitch. */
    LONG CamAngleOffsetYaw;                         /*!< Mandatory. Camera angle offset yaw. */
    ULONG EmbeddedLineNum;                          /*!< Mandatory. Embedded line number. */
    ULONG SkipFramesStill;                          /*!< Optional. */
    ULONG SkipFramesCapture;                        /*!< Optional. */
    ULONG AeFrameStore;                             /*!< Optional. */
    ULONG NoDPC;                                    /*!< Optional. */
    ULONG PDAFType;                                 /*!< Optional. */
    ULONG PDAFLineNum;                              /*!< Optional. */
    ULONG ExposeHighFpsPreviewStill;                /*!< Optional. */
    SNSR_SHUTTER_TYPE ShutterType;                  /*!< Optional */
    BOOLEAN P10bitSupport;                          /*!< Optional */
    BOOLEAN BISPSupport;                            /*!< Optional */
    ULONG Reserved[25];                             /*!< Optional. */
} SNSR_CAPS;

/*!
 * \brief Sensor caps extension
 * This extension definition allows sensor driver to extend more sensor caps but keep
 * sensor API interface compatible between ISP and sensor driver.
 * Notes: all new definition is expected to give an explicit ID number.
 */
typedef enum _SNSR_CAPS_EXT_ID
{
    EXT_CAPS_VERSION = 0,                          /*!< EXT_CAPS version, initial version is 0x1 */
} SNSR_CAPS_EXT_ID;

/*!
 * \brief Sensor caps extension type
 */
typedef enum _SNSR_CAPS_EXT_TYPE
{
    EXT_CAPS_TYPE_ULONG = 0,
    EXT_CAPS_TYPE_LONG  = 1,
} SNSR_CAPS_EXT_TYPE;

/*!
 * \brief Sensor caps extension data
 */
typedef union _SNSR_CAPS_EXT_DATA
{
    ULONG ulValue;                                /*!< ULONG value */
    LONG  lValue;                                 /*!< LONG value */
    ULONG Reserved[4];                            /*!< not used, reserved as data struct size */
} SNSR_CAPS_EXT_DATA;

/*!
 * \brief Sensor exposure and gain data structure
 */
typedef struct _SNSR_EXPOSURE
{
    USHORT FineIntegrationTime;                     /*!< Mandatory. Fine exposure time. */
    USHORT CoarseIntegrationTime;                   /*!< Mandatory. Coarse exposre time. */
    USHORT AnalogGain;                              /*!< Mandatory. Analog gain. */
    USHORT DigitalGain;                             /*!< Mandatory. Digital gain. */
    USHORT LineLengthPixels;                        /*!< Mandatory. LineLengthPixels. */  
    USHORT FrameLengthLines;                        /*!< Mandatory. Frame length lines. */
} SNSR_EXPOSURE;

/*!
* \brief Sensor exposure and gain data structure
*/
typedef struct _SENSOR_MODE_DATA
{
    unsigned int VtPixClkFreqHz;                    /*!< Mandatory. Sensor pixel clock in Hz. */    
    USHORT LineLengthPck;                           /*!< Mandatory. Sensor HTS. */
    USHORT FrameLengthLines;                        /*!< Mandatory. Sensor VTS. */
    USHORT FineIntegrationTimeMin;                  /*!< Mandatory. Sensor exposure fine min value. */
    USHORT FineIntegrationTimeMaxMargin;            /*!< Mandatory. Sensor exposure fine max margin. */
    USHORT CoarseIntegrationTimeMin;                /*!< Mandatory. Sensor exposure coarse min value. */
    USHORT CoarseIntegrationTimeMaxMargin;          /*!< Mandatory. Sensor exposure coarse max margin. */
    USHORT CropHorizontalStart;                     /*!< Mandatory. Sensor crop start cord. (x0,y0) */
    USHORT CropVerticalStart;                       /*!< Mandatory. Sensor crop start cord. (x0,y0) */
    USHORT CropHorizontal_end;                      /*!< Mandatory. Sensor crop end cord. (x1,y1) */
    USHORT CropVertical_end;                        /*!< Mandatory. Sensor crop end cord. (x1,y1) */
    USHORT OutputWidth;                             /*!< Mandatory. Sensor output size to ISP after binning/scaling */
    USHORT OutputHeight;                            /*!< Mandatory. Sensor output size to ISP after binning/scaling */
    UCHAR  BinningFactorX;                          /*!< Mandatory. Sensor horizontal binning factor used */
    UCHAR  BinningFactorY;                          /*!< Mandatory. Sensor vertical binning factor used */
    USHORT IspEffectiveInputWidth;                  /*!< Mandatory. ISP input size */
    USHORT IspEffectiveInputHeight;                 /*!< Mandatory. ISP input size */
    USHORT IspOutputWidth;                          /*!< Mandatory. ISP output size */
    USHORT IspOutputheight;                         /*!< Mandatory. ISP output size */
} SENSOR_MODE_DATA, *PSENSOR_MODE_DATA;

/*!
* \brief Sensor vcm init data data structure
*/
typedef struct _SNSR_VCMDATA
{
    ULONG Count;                                    /*!< Mandatory. Sensor vcm date number */
    struct {
        ULONG Index;                                /*!< Mandatory. Vcm setting index */
        ULONG Value;                                /*!< Mandatory. Vcm setting value */
    } data[1];
} SNSR_VCMDATA;

/*!
 * \brief SEND_INPUT_IMAGE function.
 * This function is called to send input image.
 *
 * \param[in]     stream            Mandatory.\n
 *                                  The pointer of stream.
 * \param[in]     data              Mandatory.\n
 *                                  The pointer of data.
 * \param[out]    width             Mandatory.\n
 *                                  The image width.
 * \param[out]    height            Mandatory.\n
 *                                  The image height.
 */
typedef
void                  
(* SEND_INPUT_IMAGE) (
    const void *stream,
    const unsigned short *data,                                
    unsigned int width,                                
    unsigned int height 
    );

#pragma warning( disable: 4214 )

/* \brief Sensor lock value sturcture
*/
typedef struct _SNSR_LOCK_VALUE
{
    BOOLEAN bEnabled;
    USHORT Value;
}SNSR_LOCK_VALUE;

/*!
* \brief Sensor lock params enum
*/
typedef enum _SNSR_ID_LOCK_PARAMS_VALUES
{
    SNSR_LOCK_PARAMS_FRAME_LENGTH_LINES = 0,
    SNSR_LOCK_PARAMS_COARSE_INTEGRATION,
    SNSR_LOCK_PARAMS_ANALOG_GAIN,
    SNSR_LOCK_PARAMS_DIGITAL_GAIN,
    SNSR_LOCK_PARAMS_MAX
}SNSR_ID_LOCK_PARAMS_VALUES;

/*!
* \brief Sensor lock params sturcture
*/
typedef struct _SNSR_LOCK_PARAMS
{
    SNSR_LOCK_VALUE Lock[SNSR_LOCK_PARAMS_MAX];
}SNSR_LOCK_PARAMS, *PSNSR_LOCK_PARAMS;

typedef enum __SNSR_EXT_FEATURE_PRIVACY_LED_STATE
{
    SNSR_EXT_FEATURE_PRIVACY_LED_OFF,
    SNSR_EXT_FEATURE_PRIVACY_LED_ON,
} SNSR_EXT_FEATURE_PRIVACY_LED_STATE;

//Test mode params
typedef struct _SNSR_TESTMODE_PARAMS_
{
    BOOLEAN Enabled;
    ULONG Value;
}SNSR_TESTMODE_PARAMS, *PSNSR_TESTMODE_PARAMS;

//Specify position in X,Y
typedef struct {
    ULONG dwXPos; //X Position
    ULONG dwYPos; //Y Position
}SNSR_EXT_FEATURE_ROI_POINT, *PSNSR_EXT_FEATURE_ROI_POINT;

//Specify ROI Window
typedef struct {
    SNSR_EXT_FEATURE_ROI_POINT TLPoint; //Top left
    SNSR_EXT_FEATURE_ROI_POINT BRPoint; //Bottom Right
}SNSR_EXT_FEATURE_ROI_WINDOW, *PSNSR_EXT_FEATURE_ROI_WINDOW;

//Sensor Binning mode
typedef enum {
    SNSR_EXT_FEATURE_ROI_SENSOR_BINNING_1x1 = 1,
    SNSR_EXT_FEATURE_ROI_SENSOR_BINNING_2x2,
    SNSR_EXT_FEATURE_ROI_SENSOR_BINNING_3x3,
    SNSR_EXT_FEATURE_ROI_SENSOR_BINNING_MAX
}SNSR_EXT_FEATURE_ROI_SENSOR_BINNING_VALUE, *PSNSR_EXT_FEATURE_ROI_SENSOR_BINNING_VALUE;

//ROI params
typedef struct {
    SNSR_EXT_FEATURE_ROI_SENSOR_BINNING_VALUE Binning; //Current sensor binning mode
    SNSR_EXT_FEATURE_ROI_WINDOW ActiveROI; //Active ROI
    SNSR_EXT_FEATURE_ROI_WINDOW FullROI; //Full sensor ROI
}SNSR_EXT_FEATURE_ROI_PARAMS, *PSNSR_EXT_FEATURE_ROI_PARAMS;


typedef struct
{
    UINT8 closed_loop_hall_offset_vert;             /*!< vertical distance b/w Hall sensor and Magnet */
    UINT8 closed_loop_hall_bias_horz;               /*!< horizontal distance b/w Hall sensor and Magnet */
    UINT8 pdaf_ps_sensor_modes;                     /*!< number of sensor modes calibrated for PS */
    UINT8 pdaf_ps_knots_width;                      /*!< number of PS knots across width */
    UINT8 pdaf_ps_knots_height;                     /*!< number of PS knots across height */
    UINT8 pdaf_ps_frac_bits;                        /*!< PS Fractional Bit depth */
    UINT16* pdaf_ps_tables;                         /*!< PS Gain tables for Left Pixels */
}SNSR_CALIBRATION_DATA;

#include "sensor_feature.h"
#include "sensor_skycam.h"

#endif
